' Copyright (c) 2018 Roku, Inc. All rights reserved.

' Converts associative array to a node of a given type
' @param inputAA associative array, which will be transformed to roSGNode
' @param nodeType type of node, which will be created
function Utils_AAToNode(inputAA = {} as Object, nodeType = "Node" as String) as Object
    node = createObject("roSGNode", nodeType)

    Utils_forceSetFields(node, inputAA)

    return node
end function


'converts AA to ContentNode
Function Utils_AAToContentNode(inputAA = {} as Object, nodeType = "ContentNode" as String)
    return Utils_AAToNode(inputAA, nodeType)
End Function


' Force sets fields to a given node. If node doesn't have a field, it adds it and then sets
' @param node roSGNode, to which you want to set fields
' @param fieldsToSet associative array of field names and values to be set
sub Utils_forceSetFields(node as Object, fieldsToSet as Object)
    ' if not fw_isSGNode(node) or not fw_isAssociativeArray(fieldsToSet) then return

    existingFields = {}
    newFields = {}

      'AA of node read-only fields for filtering'
    fieldsFilterAA = {
        focusedChild    :   "focusedChild"
        change          :   "change"
        metadata        :   "metadata"
    }

    for each field in fieldsToSet
        if node.hasField(field)
            if NOT fieldsFilterAA.doesExist(field) then existingFields[field] = fieldsToSet[field]
        else
            newFields[field] = fieldsToSet[field]
        end if
    end for

    node.setFields(existingFields)
    node.addFields(newFields)
end sub

'get parent based on subtype
' @param nodeName [String] parent subtype
Function Utils_getParent(nodeName as String, node = GetGlobalAA().top as Object) as Object
    
    while node <> invalid and lCase(node.subtype()) <> lCase(nodeName)
        node = node.getParent()
    end while

    return node
End Function

'get parent based on index
' @param index [Integer] parent subtype
Function Utils_getParentbyIndex(index as Integer, node = GetGlobalAA().top as Object) as Object
    
    while node <> invalid and index > 0 
        node = node.getParent()
        index--
    end while
    
    'if node <> invalid AND index = 0 
    return node
End Function

'converts array of AAs to content node with child content nodes
Function Utils_ContentList2Node(contentList as Object) as Object
    result = createObject("roSGNode","ContentNode")

    for each itemAA in contentList
        item = Utils_AAToContentNode(itemAA, "ContentNode")
        result.appendChild(item)
    end for

    return result
End Function


Function Utils_CopyNode(node as Object) as Object
    if node = invalid then return invalid
    result = createObject("roSGNode","ContentNode")
    Utils_forceSetFields(result, node.getFields())
    return result
End Function


Function Utils_DurationAsString(durationSeconds as Integer) as String
    duration = ""

    if durationSeconds > 0
        date = createObject("roDateTime")
        date.fromSeconds(durationSeconds)

        hours = date.getHours()
        minutes = date.getMinutes()
        seconds = date.getSeconds()

        if hours > 0
            duration = hours.toStr() + tr("h")
        end if

        if minutes > 0
            if isnonemptystr(duration)
                duration = duration + " "
            end if
            duration = duration + minutes.toStr() + tr("m")
        end if

        if hours = 0 AND minutes < 10
            if isnonemptystr(duration)
                duration = duration + " "
            end if
            duration = duration + seconds.toStr() + tr("s")
        end if
    end if

    return duration
End Function



REM ******************************************************
REM Copyright Roku 2011,2012,2013.
REM All Rights Reserved
REM ******************************************************

REM Functions in this file:
REM     isnonemptystr
REM     isnullorempty
REM     strtobool
REM     itostr
REM     strTrim
REM     strTokenize
REM     joinStrings
REM     strReplace
REM     

'******************************************************
'isnonemptystr
'
'Determine if the given object supports the ifString interface
'and returns a string of non zero length
'******************************************************
Function isnonemptystr(obj) As Boolean
    return ((obj <> invalid) AND (GetInterface(obj, "ifString") <> invalid) AND (Len(obj) > 0))
End Function


'******************************************************
'isnullorempty
'
'Determine if the given object is invalid or supports
'the ifString interface and returns a string of zero length
'******************************************************
Function isnullorempty(obj) As Boolean
    return ((obj = invalid) OR (GetInterface(obj, "ifString") = invalid) OR (Len(obj) = 0))
End Function


'******************************************************
'strtobool
'
'Convert string to boolean safely. Don't crash
'Looks for certain string values
'******************************************************
Function strtobool(obj As dynamic) As Boolean
    if obj = invalid return false
    if type(obj) <> "roString" and type(obj) <> "String" return false
    o = strTrim(obj)
    o = Lcase(o)
    if o = "true" return true
    if o = "t" return true
    if o = "y" return true
    if o = "1" return true
    return false
End Function

'******************************************************
'booltostr
'
'Converts a boolean value to a cannonical string value
'******************************************************
Function booltostr(bool As Boolean) As String
    if bool = true then return "true"
    return "false"
End Function

'******************************************************
'itostr
'
'Convert int to string. This is necessary because
'the builtin Stri(x) prepends whitespace
'******************************************************
Function itostr(i As Integer) As String
    str = Stri(i)
    return strTrim(str)
End Function


'******************************************************
'Trim a string
'******************************************************
Function strTrim(str As String) As String
    st=CreateObject("roString")
    st.SetString(str)
    return st.Trim()
End Function


'******************************************************
'Tokenize a string. Return roList of strings
'******************************************************
Function strTokenize(str As String, delim As String) As Object
    st=CreateObject("roString")
    st.SetString(str)
    return st.Tokenize(delim)
End Function

'******************************************************
' Joins an array or list of strings together.
' Performs the opposite function of strTokenize().
'
'@param stringArray An array or list of strings
'@param separator   The separator string to be placed between each string

'@return A single string which is the concatenation of all
'        the strings in the string array/list and spaced apart
'        by the separator string.
'******************************************************
Function joinStrings(stringArray as Object, separator as String) as String
    joinedString = ""

    addedPreviousString = false
    for each stringEntry in stringArray
        if isnonemptystr(stringEntry)
            if addedPreviousString
                joinedString = joinedString + separator
            else
                addedPreviousString = true
            end if
            joinedString = joinedString + stringEntry
        end if
    end for
 
    return joinedString
End Function

'******************************************************
'Replace substrings in a string. Return new string
'******************************************************
Function strReplace(basestr As String, oldsub As String, newsub As String) As String
    newstr = ""

    i = 1
    while i <= Len(basestr)
        x = Instr(i, basestr, oldsub)
        if x = 0 then
            newstr = newstr + Mid(basestr, i)
            exit while
        endif

        if x > i then
            newstr = newstr + Mid(basestr, i, x-i)
            i = x
        endif

        newstr = newstr + newsub
        i = i + Len(oldsub)
    end while

    return newstr
End Function

'TODO: move NormalizeURL() to Http.brs

'
' NWM 130811
' attempt to parse, decode, and re-encode a URL to fix any poorly encoded characters 
' that might cause roURLTransfer.SetURL() to fail
'
function NormalizeURL(url)
  result = url
  
  xfer = CreateObject("roURLTransfer")
  xfer.SetURL(url)
  if xfer.GetURL() = url
    ? "NormalizeURL: SetURL() succeeded, normalization not necessary"
    return result
  end if
  
  bits = url.Tokenize("?")
  if bits.Count() > 1
    result = bits[0] + "?"
    
    pairs = bits[1].Tokenize("&")
    for each pair in pairs
      keyValue = pair.Tokenize("=")

      key = xfer.UnEscape(keyValue[0])
      ? "NormalizeURL: un-escaped key " + key
      key = xfer.Escape(key)
      ? "NormalizeURL: re-escaped key " + key
      
      result = result + key

      if keyValue.Count() > 1
        value = xfer.UnEscape(keyValue[1])
        ? "NormalizeURL: un-escaped value " + value
        value = xfer.Escape(value)
        ? "NormalizeURL: re-escaped value " + value
        
        result = result + "=" + value
      end if

      result = result + "&"
    next
    
    result = result.Left(result.Len() - 1)
    ? "NormalizeURL: normalized URL " + result

    xfer.SetURL(result)
    if xfer.GetURL() = result
      ? "NormalizeURL: SetURL() succeeded with normalized URL"
    else
      ? "NormalizeURL: ***ERROR*** SetURL() failed with normalized URL"
    end if
  end if
  
  return result
end function


function RegistryUtil() as Object
    registry = {

        '** Writes value to Registry
        '@param key Registry section key
        '@param val value to write
        '@param section Registry section name
        write: function(key as String, val as String, section = "default" as String) as Void
            sec = createObject("roRegistrySection", section)
            sec.write(key, val)
            sec.flush()
        end function

        '** Reads value from Registry
        '@param key Registry section key
        '@param section Registry section name
        read: function(key as String, section = "default" as String) as Dynamic
            sec = createObject("roRegistrySection", section)
            if sec.exists(key) then return sec.read(key)
            return invalid
        end function

        '** Retrieve all entries in the specified section
        '@param section Registry section name
        readSection: function(section = "default" as String) as Object
            sec = createObject("roRegistrySection", section)
            aa = {}
            keyList = sec.getKeyList()
            for each key in keyList
                aa[key] = m.read(key, section)
            end for
            return aa
        end function

        '** Deletes key value from Registry
        '@param key Registry section key
        '@param section Registry section name
        delete: function(key as String, section = "default" as String) as Dynamic
            sec = createObject("roRegistrySection", section)
            if sec.exists(key) then return sec.delete(key)
            return invalid
        end function

        '** Deletes all key values from the specified section
        '@param section Registry section name
        deleteSection: function(section = "default" as String) as Boolean
            reg = createObject("roRegistry")
            return reg.delete(section)
        end function

        '** Deletes all sections from the registry
        '@param section Registry section name
        clear: function()
            sectionList = m.getSections()
            for each section in sectionList
                m.deleteSection(section)
            end for
        end function

        '** Retrieve all sections in the registry
        getSections: function() as Object
            reg = createObject("roRegistry")
            return reg.getSectionList()
        end function
    }

    return registry
end function


function fetch(options)
    timeout = options.timeout
    if timeout = invalid then timeout = 0

    response = invalid
    port = CreateObject("roMessagePort")
    request = CreateObject("roUrlTransfer")
    request.SetCertificatesFile("common:/certs/ca-bundle.crt")
    request.InitClientCertificates()
    request.RetainBodyOnError(true)
    request.SetMessagePort(port)
    if options.headers <> invalid
        for each header in options.headers
            val = options.headers[header]
            if val <> invalid then request.addHeader(header, val)
        end for
    end if
    if options.method <> invalid
        request.setRequest(options.method)
    end if
    request.SetUrl(options.url)

    requestSent = invalid
    if options.body <> invalid
        requestSent = request.AsyncPostFromString(options.body)
    else
        requestSent = request.AsyncGetToString()
    end if
    if (requestSent)
        msg = wait(timeout, port)
        status = -999
        body = "(TIMEOUT)"
        headers = {}
        if (type(msg) = "roUrlEvent")
            status = msg.GetResponseCode()
            headersArray = msg.GetResponseHeadersArray()
            for each headerObj in headersArray
                for each headerName in headerObj
                    val = {
                        value: headerObj[headerName]
                        next: invalid
                    }
                    current = headers[headerName]
                    if current <> invalid
                        prev = current
                        while current <> invalid
                            prev = current
                            current = current.next
                        end while
                        prev.next = val
                    else
                        headers[headerName] = val
                    end if
                end for
            end for
            body = msg.GetString()
            if status < 0 then body = msg.GetFailureReason()
        end if

        response = {
            _body: body,
            status: status,
            ok: (status >= 200 AND status < 300),
            headers: headers,
            text: function()
                return m._body
            end function,
            json: function()
                return ParseJSON(m._body)
            end function,
            xml: function()
                if m._body = invalid then return invalid
                xml = CreateObject("roXMLElement") '
                if NOT xml.Parse(m._body) then return invalid
                return xml
            end function
        }
    end if

    return response
end function


function ArrayUtil() as Object

    util = {

        isArray: function(arr) as Boolean
            return type(arr) = "roArray"
        end function

        contains: function(arr as Object, element as Dynamic) as Boolean
            return m.indexOf(arr, element) >= 0
        end function

        indexOf: function(arr as Object, element as Dynamic) as Integer
            if not m.isArray(arr) then return -1

            size = arr.count()

            if size = 0 then return -1

            for i = 0 to size - 1
                if arr[i] = element then return i
            end for

            return -1
        end function

        lastIndexOf: function(arr as Object, element as Dynamic) as Integer
            if not m.isArray(arr) then return -1

            size = arr.count()

            if size = 0 then return -1

            for i = size - 1 to 0 step -1
                if arr[i] = element then return i
            end for

            return -1
        end function

        slice: function(arr as Object, fromIndex as Integer, toIndex=invalid as Dynamic)
            if not m.isArray(arr) then return invalid

            size = arr.count()
            lastIndex = size - 1
            slicedArr = []
            
            if fromIndex < 0 then fromIndex = size + fromIndex
            if toIndex = invalid then toIndex = lastIndex
            if toIndex < 0 then toIndex = size + toIndex
            if toIndex >= size then toIndex = lastIndex

            if fromIndex >= size OR fromIndex > toIndex then return slicedArr

            for i = fromIndex to toIndex
                slicedArr.push(arr[i])
            end for

            return slicedArr
        end function

        map: function(arr as Object, func as Function)
            if not m.isArray(arr) then return invalid

            size = arr.count()
            mappedArr = []

            if size = 0 then return mappedArr

            for i = 0 to size - 1
                mappedArr.push(func(arr[i], i, arr))
            end for

            return mappedArr
        end function

        reduce: function(arr as Object, func as Function, initialValue=invalid as Dynamic)
            if not m.isArray(arr) then return invalid

            size = arr.count()
            startAt = 0
            accumulator = initialValue

            if size = 0 then return accumulator

            if accumulator = invalid then 
                accumulator = arr[0]
                startAt = 1
            end if

            for i = startAt to size - 1
                accumulator = func(accumulator, arr[i], i, arr)
            end for
            
            return accumulator
        end function

        filter: function(arr as Object, func as Function)
            if not m.isArray(arr) then return invalid

            size = arr.count()
            mappedArr = []

            if size = 0 then return mappedArr

            for i = 0 to size - 1
                if func(arr[i], i, arr) then
                    mappedArr.push(arr[i])
                end if
            end for

            return mappedArr
        end function

    }

    return util

end function

Function HandleFav(id as string)
    fav = RegistryUtil().read("fav",m.global.manifest.key)
    if fav = invalid
        fav = []
    else 
        fav = ParseJson(fav)    
    end if
    if ArrayUtil().contains(fav,id)
        index = ArrayUtil().indexOf(fav,id)
        fav.delete(index)
    else
       fav.push(id)    
    end if

    fav = FormatJson(fav)
    RegistryUtil().write("fav",fav,m.global.manifest.key)  
End Function

Function GetFavIDS()
    fav = RegistryUtil().read("fav",m.global.manifest.key)
    if fav = invalid
        fav = []
    else 
        fav = ParseJson(fav)    
    end if
    return fav
end function

Function GetFavData()
    fav = GetFavIDS()
    favdata = []
    if fav.count() > 0
        for each channel in m.global.channels
            if ArrayUtil().contains(fav,channel.stream_id.toStr())
                favdata.push(channel)
            end if
        end for
    end if
    return favdata
End Function


Function GetRecentIDS()
    rec = RegistryUtil().read("recent",m.global.manifest.key)
    if rec = invalid
        rec = []
    else 
        rec = ParseJson(rec)    
    end if
    return rec
end function


Function GetRecentData()
    rec = GetRecentIDS()
    recdata = []
    if rec.count() > 0
        for each channel in m.global.channels
            if ArrayUtil().contains(rec,channel.stream_id) and ArrayUtil().contains(m.global.adultID,channel.category_id) = false
                recdata.push(channel)
            end if
        end for
    end if
    recdata.Reverse() 
    return recdata
End Function

Function GetChannelIndex(data as object,channel as object)
    if data.count() > 0
        i = 0
        for each dataItem in data
            if dataItem.stream_id = channel.stream_id or dataItem.stream_id = channel.stream_id
                return i
            end if
            i = i + 1
        end for
    else
        return invalid
    end if
    return invalid
End Function


Function HandleRecent(id as string)
    rec = GetRecentIDS()
    if rec = invalid
        rec = []
    end if

    if ArrayUtil().contains(rec,id)
        index = ArrayUtil().indexOf(rec,id)
        rec.delete(index)
        rec.push(id)
    else
        rec.push(id)    
    end if
    
    if rec.count() > 15
        rec.Shift()
    end if
    rec = FormatJson(rec)
    RegistryUtil().write("recent",rec,m.global.manifest.key)  
End Function


Function GetCurrentEPG(data as object)
    date = CreateObject("roDateTime")
    items = []
    yes = false
    if data <> invalid and data.count() > 0
        CurrentTime = date.AsSeconds()
        for each dataItem in data
            if (dataItem.start_timestamp.toint() < CurrentTime and dataItem.stop_timestamp.toint() > CurrentTime) or (yes=true)
                items.push(dataItem)
                yes = true
                if items.count() = 3
                    return items
                end if
            end if
        end for
    end if
    return invalid
End function

Function GetEPGChannel(data as object)
    i = 0
    for each dataItem in data
        if type(dataItem.epg_channel_id) <> "Invalid"
            i = i + 1
            return i
        end if
    end for
    return i
End function


Function GetDurationString(totalSeconds = 0 As Integer) As String
    remaining = totalSeconds
    hours = Int(remaining / 3600).ToStr()
    remaining = remaining Mod 3600
    minutes = Int(remaining / 60).ToStr()
    remaining = remaining Mod 60
    seconds = remaining.ToStr()

    If hours <> "0" Then
        Return PadLeft(hours, "0", 2) + "hr " + PadLeft(minutes, "0", 2) + "min "
    Else
        Return PadLeft(minutes, "0", 2) + "min"
    End If
End Function

Function PadLeft(value As String, padChar As String, totalLength As Integer) As String
    While value.Len() < totalLength
        value = padChar + value
    End While
    Return value
End Function

Function GetMessagesIDS()
    message = RegistryUtil().read("messageIDS",m.global.manifest.key)
    if message = invalid
        message = []
    else 
        message = ParseJson(message)    
    end if
    return message
End function


Function CheckNewMessage()
    ids = GetMessagesIDS()
    ? ids
    inCheck = false
    messages = RegistryUtil().read("message",m.global.manifest.key)
    if messages <> invalid
        messages = ParseJson(messages)
    else
        return inCheck   
    end if
    for each message in messages
       if not ArrayUtil().contains(ids,message.messageid)
        inCheck = true
       end if
    end for
    return inCheck
End Function


Function HandleMessages(id as string)
    message = RegistryUtil().read("messageIDS",m.global.manifest.key)
    if message = invalid
        message = []
    else 
        message = ParseJson(message)    
    end if

    if ArrayUtil().contains(message,id) = false
        message.push(id)
    end if

    message = FormatJson(message)    
    RegistryUtil().write("messageIDS",message,m.global.manifest.key)  
End Function



Function HandleFavAssets(data as object,AType as string)
    fav = RegistryUtil().read(AType,m.global.manifest.key)
    if fav = invalid
        fav = []
    else 
        fav = ParseJson(fav)    
    end if

    present = false

    i = 0
    index = 0
    for each asset in fav
        if AType = "TVShows"
            if data.series_id = asset.series_id
                present = true
                index = i
            end if
        else
            if data.stream_id = asset.stream_id
                present = true
                index = i
            end if
        end if
        i = i + 1
    end for
    if present = true
        fav.delete(index)
    else
       fav.push(data)    
    end if

    fav = FormatJson(fav)
    RegistryUtil().write(AType,fav,m.global.manifest.key)  
End Function


Function GetFavAssets(data as object,AType as string)
    fav = RegistryUtil().read(AType,m.global.manifest.key)
    if fav = invalid
        fav = []
    else 
        fav = ParseJson(fav)    
    end if

    present = false

    for each asset in fav
        if AType = "TVShows"
            if data.series_id = asset.series_id
                return true
            end if
        else
            if data.stream_id = asset.stream_id
                return true
            end if
        end if
    end for
    return false
End Function


Function GetFavAssetsList(AType as string)
    fav = RegistryUtil().read(AType,m.global.manifest.key)
    if fav = invalid
        fav = []
    else 
        fav = ParseJson(fav)    
    end if
    return fav
End Function



Function HandleFields(key as string,data as object,content as object)
    if content.hasfield(key)
        content.setField(key, data)
    else
        if type(data) = "roAssociativeArray"
            dataType = "assocarray"
        else 
            dataType = "array"
        end if
        content.addField(key, dataType,true)
        content.setField(key, data)
    end if
    if content.hasfield("fieldchanged")
        content.setField("fieldchanged", key)
    else
        content.addField("fieldchanged", "string", true)
        content.setField("fieldchanged", key)
    end if
end Function



Function GetTheme(id as string)
    if m.global.Cdata <> invalid and m.global.Cdata.app_mode <> invalid
        data = m.global.Cdata
    else if m.global.Udata <> invalid and m.global.Udata.app_mode <> invalid
        data = m.global.Udata
    end if

    if id = "roku_button_unfocus"
        if data.themes.roku_button_unfocus <> invalid and data.themes.roku_button_unfocus <> ""
            return data.themes.roku_button_unfocus
        else
            return "#737373"
        end if
    else if id = "roku_button_focus"
        if data.themes.roku_button_focus <> invalid and data.themes.roku_button_focus <> ""
            return data.themes.roku_button_focus
        else
            return "#d42e00"
        end if
    else if id = "roku_color_primary"
        if data.themes.roku_color_primary <> invalid and data.themes.roku_color_primary <> ""
            return data.themes.roku_color_primary
        else
            return "#281151"
        end if
    else if id = "roku_color_secondary"
        if data.themes.roku_color_secondary <> invalid and data.themes.roku_color_roku_color_secondaryprimary <> ""
            return data.themes.roku_color_secondary
        else
            return "#281151"
        end if    
    else if id = "roku_background_overlay"
        if data.themes.roku_background_overlay <> invalid and data.themes.roku_background_overlay <> ""
            return data.themes.roku_background_overlay
        else
            return "#FFFFFF00"
        end if
    end if
End function